﻿import units
import math

context = ExtAPI.Context
if context == "Mechanical":
    link = {}
    #kHex20
    link.Add(ElementTypeEnum.kHex20,{ 0:[3,1,4], 1:[0,2,5], 2:[1,3,6], 3:[2,0,7], 4:[5,0,7], 5:[1,4,6], 6:[5,7,2], 7:[6,3,4], 8:[0,1], 9:[1,2], 10:[2,3], 11:[3,0], 12:[4,5], 13:[5,6], 14:[6,7], 15:[7,4], 16:[0,4], 17:[1,5], 18:[2,6], 19:[3,7]})
    #kHex8
    link.Add(ElementTypeEnum.kHex8,{ 0:[3,1,4], 1:[0,2,5], 2:[1,3,6], 3:[2,0,7], 4:[5,0,7], 5:[1,4,6], 6:[5,7,2], 7:[6,3,4]})
    #kPyramid13
    link.Add(ElementTypeEnum.kPyramid13,{ 0:[3,1,4], 1:[0,2,4], 2:[1,3,4], 3:[2,0,4], 4:[0,1,2,3], 5:[0,1], 6:[1,2], 7:[2,3], 8:[3,0], 9:[0,4], 10:[1,4], 11:[2,4], 12:[3,4]})
    #kPyramid5
    link.Add(ElementTypeEnum.kPyramid5,{ 0:[3,1,4], 1:[0,2,4], 2:[1,3,4], 3:[2,0,4], 4:[0,1,2,3]})
    #kQuad4
    link.Add(ElementTypeEnum.kQuad4, { 0:[3,1], 1:[0,2], 2:[1,3], 3:[2,0]})
    #kQuad8
    link.Add(ElementTypeEnum.kQuad8, { 0:[3,1], 1:[0,2], 2:[1,3], 3:[2,0], 4:[0,1], 5:[1,2], 6:[2,3], 7:[3,0]})
    #kTet10
    link.Add(ElementTypeEnum.kTet10,{ 0:[2,1,3], 1:[0,2,3], 2:[1,0,3], 3:[0,1,2], 4:[0,1], 5:[1,2], 6:[2,0], 7:[0,3], 8:[1,3], 9:[2,3]})
    #kTet4
    link.Add(ElementTypeEnum.kTet4, { 0:[2,1,3], 1:[0,2,3], 2:[1,0,3], 3:[0,1,2]})
    #kTri3
    link.Add(ElementTypeEnum.kTri3, { 0:[2,1], 1:[0,2], 2:[1,0]})
    #kTri6
    link.Add(ElementTypeEnum.kTri6, { 0:[2,1], 1:[0,2], 2:[1,0], 3:[0,1], 4:[1,2], 5:[2,0]})
    #kWedge15
    link.Add(ElementTypeEnum.kWedge15,{ 0:[2,1,3], 1:[0,2,4], 2:[1,0,5], 3:[5,4,0], 4:[3,5,1], 5:[4,3,2], 6:[0,1], 7:[1,2], 8:[2,0], 9:[3,4], 10:[4,5], 11:[5,3], 12:[0,3], 13:[1,4], 14:[2,5]})
    #kWedge6
    link.Add(ElementTypeEnum.kWedge6,{ 0:[2,1,3], 1:[0,2,4], 2:[1,0,5], 3:[5,4,0], 4:[3,5,1], 5:[4,3,2]})


def Create_Mises_Result(analysis):
    analysis.CreateResultObject("Von Mises Stress", ExtAPI.ExtensionManager.CurrentExtension)


mises_stress = {}
    
# This function evaluates the specific result (i.e. the Von-Mises stress) on each element required by the geometry selection
# The input data "step" represents the step on which we have to evaluate the result
def Mises_At_Nodes_Eval(result, step):
    global mises_stress
    
    ExtAPI.Log.WriteMessage("Launch evaluation of the Mises result at nodes: ")
    # Reader initialization
    reader = result.Analysis.GetResultsData()
    istep = int(step)
    reader.CurrentResultSet = istep
    # Get the stress result from the reader
    stress = reader.GetResult("S")
    stress.SelectComponents(["X", "Y", "Z", "XY", "XZ", "YZ"])
    unit_stress = stress.GetComponentInfo("X").Unit
    conv_stress = units.ConvertUnit(1.,unit_stress,"Pa","Stress") 
    
    # Get the selected geometry 
    propGeo = result.Properties["Geometry"]
    refIds = propGeo.Value.Ids
    
    nodal_stress = [0.] * 6
    
    # Get the mesh of the model
    mesh = result.Analysis.MeshData
    
    # Loop on the list of the selected geometrical entities
    for refId in refIds:
        # Get mesh information for each geometrical entity
        meshRegion = mesh.MeshRegionById(refId)
        nodeIds = meshRegion.NodeIds
        
        # Loop on the nodes related to the current geometrical refId
        for nodeId in nodeIds:
            
            for i in range(6):
                nodal_stress[i] = 0.
                
            elementIds = mesh.NodeById(nodeId).ConnectedElementIds
            num = 0
            # Loop on the elements related to the current node
            for elementId in elementIds:
                # Get the stress tensor related to the current element
                tensor = stress.GetElementValues(elementId)
                
                element = mesh.ElementById(elementId)
                # Look for the position of the node nodeId in the element elementId
                # cpt contains this position
                cpt = element.NodeIds.IndexOf(nodeId)
                
                # for corner nodes, cpt is useless. 
                # The n corner nodes of one element are always the first n nodes of the list 
                if cpt < element.CornerNodeIds.Count:
                    for i in range(6):
                        nodal_stress[i] = nodal_stress[i] + tensor[6*cpt+i]
                else:       
                # For midside nodes, cpt is used and the link table provides the two neighbouring corner nodes for the midside node identified in the list by cpt.            
                    itoadd = link[element.Type][cpt]
                    for ii in itoadd:
                        for i in range(6):
                            nodal_stress[i] = nodal_stress[i] + tensor[6*ii+i] / 2.
                
                num += 1
            
            # The average stress tensor is computed before to compute the Von-Mises stress
            # num is the number of elements connected with the current node nodeId  
            for i in range(6):
                nodal_stress[i] *= conv_stress / num
            
            # Von-Mises stress computation
            vm_stress = Mises(nodal_stress)
            # Result storage
            mises_stress[nodeId] = [vm_stress]
                

# This function returns the values array. This array will be used by Mechanical to make the display available                 
def Mises_At_Nodes_GetValue(result,nodeId):
    global mises_stress
    try:    return mises_stress[nodeId]
    except: return [System.Double.MaxValue]
    

# This function computes the Von-Mises stress from the stress tensor
# The Von-Mises stess is computed based on the three eigenvalues of the stress tensor 
def Mises(tensor):

    # Computation of the eigenvalues 
    (S1, S2, S3) = EigenValues(tensor)
    return sqrt( ( (S1-S2)*(S1-S2) + (S2-S3)*(S2-S3) + (S1-S3)*(S1-S3) ) / 2. )


# This function computes the three eigenvalues of one [3*3] symetric tensor
EPSILON = 1e-4
def EigenValues(tensor):
 
    a = tensor[0]
    b = tensor[1]
    c = tensor[2]
    d = tensor[3]
    e = tensor[4]
    f = tensor[5]
    
    if ((abs(d)>EPSILON) or (abs(e)>EPSILON) or (abs(f)>EPSILON)):
        # Polynomial reduction
        A = -(a+b+c)
        B = a*b+a*c+b*c-d*d-e*e-f*f
        C = d*d*c+f*f*a+e*e*b-2*d*e*f-a*b*c

        p = B-A*A/3
        q = C-A*B/3+2*A*A*A/27
        R = sqrt(fabs(p)/3)
        if q < 0: R = -R
    
        z = q/(2*R*R*R)
        if z < -1. : z = -1.
        elif z > 1.: z = 1.
        phi = acos(z)

        S1 = -2*R*cos(phi/3)-A/3
        S2 = -2*R*cos(phi/3+2*math.pi/3)-A/3
        S3 = -2*R*cos(phi/3+4*math.pi/3)-A/3
    else:
        S1 = a
        S2 = b
        S3 = c
    
    return (S1, S2, S3)    
