import clr
import os
clr.AddReference("Ans.Utilities")
clr.AddReference("Ans.UI.Toolkit")
clr.AddReference("Ans.UI.toolkit.Base")
clr.AddReference("Ans.ProjectSchematic")
clr.AddReference("Ans.UI")
import Ansys.UI
import Ansys.ProjectSchematic 

import System
import sys
import subprocess

from Ansys.Utilities import *
from System.IO import *
from System.Diagnostics import *
from Ansys.UI.Toolkit import *

from shutil import copyfile

def onDesignPointChange(sender, args):
    """
        Called when user sets a new design point as current
    """
    if args.ActionState == Ansys.ProjectSchematic.Events.ActionState.Completed:
        if isDP0() == True:
            tasks = ExtAPI.DataModel.Tasks
            for task in tasks:
                if task.Name.Contains("GeometryCreator"):
                    # retrieving file , for dm, the file has to be in a retaine dp folder(bug), in the same folder  
                    filePath = task.Properties["GeometryFilePath"].Value
                    fileName = filePath.Split("\\")[-1]
                    SourcePath = System.IO.Path.Combine(task.ActiveDirectory,"backup_"+fileName)
                    TargetPath = System.IO.Path.Combine(task.ActiveDirectory,fileName)
                    if System.IO.File.Exists(SourcePath) == True:
                        copyfile(SourcePath, TargetPath)
                    isRegistered = IsFileRegistered(FilePath=TargetPath)
                    if isRegistered == True:
                        geoFile = GetRegisteredFile(TargetPath)
                        #Transfer file downstreams
                        outputData = task.OutputData
                        data = outputData["FEMSetup"]
                        FEMSetup = data[0]
                        FEMSetup.FEModelerFile = geoFile

def blockTransfers(sourceTemplate, targetTemplate, usableTypes):
    if sourceTemplate.Name == "GeometryCreatorTemplate":
        if targetTemplate.Name == "GeometryCellTemplate":
            return usableTypes
        else:
            return System.Collections.Generic.List[System.String]()
    return usableTypes

def getDPName():
    """
        Method to get current design point name
    """
    currDP = Parameters.GetActiveDesignPoint()
    return "dp"+currDP.Name

def isDP0():
    name = getDPName()
    return name == "dp0"

handler = None
def init(entity):
    """
        Called on opening of WB (when extension is loaded)
    """
    ExtAPI.Log.WriteMessage('initializing Custom Geometry')
    if Ansys.UI.UIManager.IsRunningInteractively:
        Ansys.UI.UIManager.Instance.InvokeOperationAsync("InstallCheck", checkFEMSetup, True)
    global handler
    handler = System.EventHandler[Ansys.ProjectSchematic.Events.DesignPointChangedEventArgs](onDesignPointChange)
    Ansys.ProjectSchematic.Events.DesignPointEventSource.GetEventSource().DesignPointChanged += handler

def getCurrentFile(task):
    """
        Return file associated with task
    """
    outputData = task.OutputData
    keyList = System.Collections.Generic.List[System.String](outputData.Keys)
    MyKey = keyList[0] # I know their is only one key
    data = outputData[MyKey]
    MyData = data[0]
    if MyKey == "FEMSetup":
        geoFile = MyData.FEModelerFile
    else:
        geoFile = MyData.TransferFile
    if geoFile != None:
        geoFile = GetDesignPointFile(geoFile)
    return geoFile

def cleanupGeomFile(task):
    """
        remove file from project
    """
    geomFile = getCurrentFile(task)
    if geomFile != None:
        task.UnregisterFile(geomFile)

def terminate(context):
    global handler
    ExtAPI.Log.WriteMessage("Terminating Custom Geometry")
    if handler != None:
        Ansys.ProjectSchematic.Events.DesignPointEventSource.GetEventSource().DesignPointChanged -= handler

import xml.etree.ElementTree as ET

def checkFEMSetup():
    defaultConfig = Ansys.Utilities.ApplicationConfiguration.DefaultConfiguration
    softwarePath = defaultConfig.WorkbenchInstallRootDirectoryPath
    platform = defaultConfig.Platform
    path=System.IO.Path.Combine(System.IO.Path.Combine(System.IO.Path.Combine(System.IO.Path.Combine(System.IO.Path.Combine(softwarePath,"Addins"),"ACT"),"bin"),platform),"TransferTypeDirectPropertyAccessList.xml")
    f =open(path,'r')
    if f.read().Contains("FEMSetup"):
        f.close()
        return
    else:
        f.close()
        res = Ansys.UI.Toolkit.MessageBox.Show(Ansys.UI.UIManager.Instance.MainWindow.Window, 
            'The Extension has detected the first launch of this App.  Proceed with initialization?', 
            'Extension Setup', 
            Ansys.UI.Toolkit.MessageBoxType.Question, 
            Ansys.UI.Toolkit.MessageBoxButtons.YesNo)
        if res == Ansys.UI.Toolkit.DialogResult.Yes:
            tree = ET.parse(path)
            root = tree.getroot()
            elem=ET.SubElement(root,"TransferType")
            elem.text = "FEMSetup"
            tree.write(path)
            Ansys.UI.Toolkit.MessageBox.Show(Ansys.UI.UIManager.Instance.MainWindow.Window, 
            'Initialization has finished.\nPlease restart Workbench to correctly use this App.', 
            'Workbench Restart Required', 
            Ansys.UI.Toolkit.MessageBoxType.Warning, 
            Ansys.UI.Toolkit.MessageBoxButtons.OK)

def updateGeometry(task):
    """
        Called when task GeometryCreator is updated.
        It creates the geometry with correct Inner Diameterss
    """
    ExtAPI.Log.WriteMessage("Updating " + task.Name)
    innerDiameter = task.Properties["Inputs/InnerDiameter"].Value
    #clean geometry if one is already present
    cleanupGeomFile(task)
    geomFileName = "Geom.stp"

    # get directory of the task
    activeDir = task.ActiveDirectory
    extensionDir = ExtAPI.Extension.InstallDir
    geomFilePath = Path.Combine(activeDir, geomFileName)
    task.Properties["GeometryFilePath"].Value = geomFilePath
    
    # retrieve path to workbench executable
    # in this exemple the geometry is generated by a journal of Workbench which uses Design Modeler
    # but the idea is to run a journal from an external software that doesn't have current parametric study support in Workbench
    # (internal program the generate geometry model, external CAD modeling software such as CREO, NX, CATIA, Solidworks, Autodesk)
    softwarePath = Ansys.Utilities.ApplicationConfiguration.DefaultConfiguration.WorkbenchInstallRootDirectoryPath
    softwareBin = System.IO.Path.Combine(softwarePath, r"Framework\bin")
    softwarePlatform = System.IO.Path.Combine(softwareBin, Ansys.Utilities.ApplicationConfiguration.DefaultConfiguration.Platform)
    softwareDir = System.IO.Path.Combine(softwarePlatform, "AnsysFW.exe")
    
    # build journal to be read in Workbench
    journal = buildjournal(task, extensionDir, activeDir, geomFileName, innerDiameter)
    exitCode = ExtAPI.ProcessUtils.Start(softwareDir, "-R", journal)
    if exitCode != 0:
        raise Exception ('Geometry Creation failed!')
    geoFile = None
    if File.Exists(geomFilePath) == False:
        raise Exception ('Geometry Creation failed!')
    else:
        if isDP0() == True:
            targetFilePath = System.IO.Path.Combine(activeDir,"backup_"+geomFileName)
            ExtAPI.Log.WriteMessage("copying\n"+geomFilePath+"\nto\n"+targetFilePath)
            copyfile(geomFilePath,targetFilePath)
        else:
            dpName = getDPName()
            targetFilePath = System.IO.Path.Combine(activeDir.replace(dpName, "dp0"),geomFileName)
            if System.IO.File.Exists(targetFilePath) == True:
                copyfile(geomFilePath,targetFilePath)
        isRegistered = IsFileRegistered(FilePath=geomFilePath)
        ExtAPI.Log.WriteMessage(geomFilePath)
        if isRegistered == True:
            geoFile = GetRegisteredFile(geomFilePath)
        else:
            geoFile = task.RegisterFile(geomFilePath)
    #Transfer file downstreams
    if isDP0() == False:
        geoFile = GetDesignPointFile(geoFile, TargetDesignPointDirectoryName="dp0")
    data = task.OutputData["FEMSetup"]
    FEMSetup = data[0]
    FEMSetup.FEModelerFile = geoFile

def buildjournal(task, extDir, projDir, geomFileName, dimension):
    """
        building journal to be run in WB
    """
    # Get the journal template from the extension directory.
    ExtAPI.Log.WriteMessage("Creating batch run journal.")
    templateFilePath = System.IO.Path.Combine(extDir,'wbjnTemplate.tmpl')
    journalFilePath = System.IO.Path.Combine(projDir, "batchrun.wbjn")
    # Create substitution Path strings.
    resourceDir = Path.Combine(extDir, "resources")
    baseGeomFilePath = Path.Combine(resourceDir, "washer.agdb")
    exportGeomFilePath = Path.Combine(projDir, geomFileName)
    # replacing information about geometry in wbjn journal file
    ExtAPI.Log.WriteMessage("Substituting %GEOMFILEPATH% with = "+ baseGeomFilePath)
    ExtAPI.Log.WriteMessage("Substituting %EXPORTGEOMETRYFILEPATH% with = "+ exportGeomFilePath)
    ExtAPI.Log.WriteMessage("Substituting %INNERDIAMQTY% with = "+ str(dimension))
    exportGeomFilePath = exportGeomFilePath.replace("\\","\\\\\\\\")
    inStream = open(templateFilePath, 'r')
    buffer = inStream.read()
    buffer = buffer.replace('%GEOMFILEPATH%',baseGeomFilePath)
    buffer = buffer.replace('%EXPORTGEOMETRYFILEPATH%',exportGeomFilePath)
    buffer = buffer.replace('%INNERDIAMQTY%', str(dimension))
    outStream = open(journalFilePath, 'w')
    outStream.write(buffer)
    inStream.close()
    outStream.close()
    return(journalFilePath.replace("\\", "\\\\"))


def isInnerDiameterValid(task,property):
    """
        Check if the input is valid.
        The inner diameter has to be less than than Outer Diameter
    """
    quantity = property.Value 
    if property.Value.GetType() == System.String:
        InnerDiameter = float(quantity.Split(" ")[0])
    else:
        InnerDiameter = property.Value.Value
    if InnerDiameter < 0 :
        return False
    elif InnerDiameter > 34:
        return False
    else:
        return True

def isFileValid(task, property):
    return True

def resetGeometry(task):
    cleanupGeomFile(task)