"""
    This python file contains functions for CADFileBridge taskgroup
"""

import System
import clr
clr.AddReference("Ans.DesignModeler.GUI")
import Ansys.DesignModeler.GUI
clr.AddReference("Ans.UI.Toolkit")
clr.AddReference("Ans.UI.Toolkit.Base")
from Ansys.UI.Toolkit import *
clr.AddReference("Ans.UI")
import Ansys.UI
from shutil import copyfile


def setGeometry(task):
    """
        This method is called when the task Geometry Bridge is updated
        It enables geometry file transfer to a component of Mecanical
        A specific treatment has to be made in order to fully support the parametric study
        For DM the file folder doesn't change (for imported geometry)
        If DM is not opened the file is in dpX folder.  This shows up
        when users set DPs as current with retained data.
    """
    filePath = filePath = task.Properties["GeometryFilePath"].Value
    if filePath is None :
        raise(Exception("You must have a geometry"))
    ExtAPI.Log.WriteMessage('Update CAD file bridge')
    container = task.InternalObject
    ActiveDir = container.GetActiveDirectory()
    fileName = filePath.Split("\\")[-1]
    targetActiveFilePath = System.IO.Path.Combine(ActiveDir,fileName)

    ExtAPI.Log.WriteMessage("copying\n"+filePath+"\nto\n"+targetActiveFilePath)

    copyfile(filePath,targetActiveFilePath)

    if isDP0() == True:
        targetFilePath = System.IO.Path.Combine(ActiveDir,"backup_"+fileName)
        ExtAPI.Log.WriteMessage("copying\n"+filePath+"\nto\n"+targetFilePath)
        copyfile(filePath,targetFilePath)
    else:
        dpName = getDPName()
        targetFilePath = System.IO.Path.Combine(ActiveDir.replace(dpName, "dp0"),fileName)
        if System.IO.File.Exists(targetFilePath) == True:
            ExtAPI.Log.WriteMessage("copying\n"+filePath+"\nto\n"+targetFilePath)
            copyfile(filePath,targetFilePath)

    geoFile = None

    if System.IO.File.Exists(targetActiveFilePath) == False:
        ExtAPI.Log.WriteMessage("GEOMETRY FILE DOES NOT EXIST!")
    else:
        isRegistered = IsFileRegistered(FilePath=targetActiveFilePath)
        ExtAPI.Log.WriteMessage(targetActiveFilePath)
        if isRegistered == True:
            geoFile = GetRegisteredFile(targetActiveFilePath)
        else:
            geoFile = RegisterFile(FilePath=targetActiveFilePath)
            AssociateFileWithContainer(geoFile, container) 
    #Transfer file downstreams
    if isDP0() == False:
        geoFile = GetDesignPointFile(geoFile, TargetDesignPointDirectoryName="dp0")
    outputData = container.GetOutputData()
    data = outputData["FEMSetup"]
    FEMSetup = data[0]
    FEMSetup.FEModelerFile = geoFile

def setFileFromGUI(task):
    """
        In this method we allow user to import a CAD file
        A popup window filters only CAD file
    """
    ExtAPI.Log.WriteMessage('gui op')
    #not do if connected to a Geometry Creation task
    for sourceTask in task.SourceTasks:
        if sourceTask.Caption =="Geometry Creation":
            msgString = "You can't import a geometry when CAD File Bridge is linked to a Geometry Creator!"
            MessageBox.Show(Ansys.UI.UIManager.Instance.MainWindow.Window, msgString, "Warning", MessageBoxType.Warning, MessageBoxButtons.OK)
            return
    #step 1 - display file dialog selection with CAD filter
    defaultIndex = 0
    filterDescriptions = ""
    context = ExtAPI.DataModel.Context
    dmAddinType = clr.GetClrType(Ansys.DesignModeler.GUI.Addin)
    filterMethod = dmAddinType.GetMethod("getImportFilters", System.Reflection.BindingFlags.Static | System.Reflection.BindingFlags.NonPublic)
    args = System.Collections.Generic.List[System.Object]()
    args.Add(context)
    args.Add(filterDescriptions)
    args.Add(defaultIndex)
    args = args.ToArray()
    filterMethod.Invoke(None, args)
    filterDescriptions = args[1]
    defaultIndex = args[2]
    filePath = ""
    selectedPathRef = clr.Reference[System.String]()
    dr = FileDialog.ShowOpenDialog(Ansys.UI.UIManager.Instance.MainWindow.Window, None, filterDescriptions, defaultIndex, selectedPathRef)
    filePath = selectedPathRef.Value
    if dr == DialogResult.OK:
        if System.String.IsNullOrEmpty(filePath) == False and System.IO.File.Exists(filePath):
            ExtAPI.Log.WriteMessage('valid file = ' + filePath)
            task.Properties["GeometryFilePath"].Value = filePath
            container=task.InternalObject
            cleanupGeomFile(container)

def resetGeometry(task):
    container = task.InternalObject
    ExtAPI.Log.WriteMessage('reset')
    task.Properties["GeometryFilePath"].Value = ""
    cleanupGeomFile(container)

def refreshGeometry(task):
    ExtAPI.Log.WriteMessage('refresh')
    container = task.InternalObject
    if task.SourceTasks.Count == 0: # No Geometry Creator linked
        #Check if a file is associated
        Files = container.GetFiles()
        if Files.Count != 0 :
            File = Files[0] # I know there is only one file
            UnregisterFile(File, DeleteFileIfNonShared=True, BackUp=True)
    else:
        #obtain input data
        upstreamData = container.GetInputDataByType(InputType="ACTGeometry")
        geoFileRef = None
        upstreamDataCount = upstreamData.Count
        if upstreamDataCount > 0:
            geoFileRef = upstreamData[0] 
            task.Properties["GeometryFilePath"].Value = geoFileRef.Location
        #unregister oldfile if present.
        cleanupGeomFile(container)
