"""
    This python file contains functions for XYZ Geometry taskgroup
"""

import clr
import os
clr.AddReference("Ans.Utilities")
clr.AddReference("Ans.UI.Toolkit")
clr.AddReference("Ans.UI.toolkit.Base")
import System
import subprocess

from Ansys.Utilities import *
from System.IO import *
from System.Diagnostics import *
from Ansys.UI.Toolkit import *

def updateGeometry(task):
    """
        Called when task GeometryCreator is updated.
        It creates the geometry with correct Inner Diameters
    """
    InnerDiameter = task.Properties["Inputs/InnerDiameter"].Value
    # check validity of inner diameter
    # if it is lower than outer diameter the update isn't done
    # necessary for parametric study
    if InnerDiameter.GetType() != System.String:
        if InnerDiameter.Value < 0 :
            #by raising an exception the update is not set to up to date
            raise(Exception("Inner Diameter must be positive")) 
            return 
        elif InnerDiameter.Value > 34:
            raise(Exception("Inner Diameter must be less than 34"))
            return 

    container = task.InternalObject
    #clean geometry if one is already present
    cleanupGeomFile(container)
    geomFileName = "Geom.stp"

    # get directory of the task
    activeDir = container.GetActiveDirectory()
    extensionDir = ExtAPI.ExtensionManager.CurrentExtension.InstallDir
    geomFilePath = Path.Combine(activeDir, geomFileName)
    
    # retrieve path to workbench executable
    # in this exemple the geometry is generated by a journal of Workbench which uses Design Modeler
    # but the idea is to run a journal from an external software that doesn't have current parametric study support in Workbench
    # (internal program the generate geometry model, external CAD modeling software such as CREO, NX, CATIA, Solidworks, Autodesk)
    softwarePath = Ansys.Utilities.ApplicationConfiguration.DefaultConfiguration.WorkbenchInstallRootDirectoryPath
    softwareBin = System.IO.Path.Combine(softwarePath, r"Framework\bin")    
    softwarePlatform = System.IO.Path.Combine(softwareBin, Ansys.Utilities.ApplicationConfiguration.DefaultConfiguration.Platform)
    softwareDir = System.IO.Path.Combine(softwarePlatform, "AnsysFW.exe")
    
    # build journal to be read in Workbench
    journal = buildjournal(container, extensionDir, activeDir, geomFileName, InnerDiameter)
    #  check if it is run from Linux
    runInMono = Ansys.Utilities.ApplicationConfiguration.DefaultConfiguration.IsRuntimeMono
    info = None
    if runInMono:
        info = None
        return
    else:
        info = System.Diagnostics.ProcessStartInfo(softwareDir, System.String.Format('{0} "{1}"', "-r", journal))
    if info != None:
        info.CreateNoWindow = True
        info.WindowStyle = System.Diagnostics.ProcessWindowStyle.Minimized
        info.UseShellExecute = False
        p = System.Diagnostics.Process.Start(info)
        p.WaitForExit()
    geoFile = None
    if File.Exists(geomFilePath) == False:
        ExtAPI.Log.WriteMessage("NO GEOMETRY FILE GENERATED!")
        return
    else:
        isRegistered = IsFileRegistered(FilePath=geomFilePath)
        if isRegistered == True:
             geoFile = GetRegisteredFile(geomFilePath)
        else:
             geoFile = RegisterFile(FilePath=geomFilePath)
             AssociateFileWithContainer(geoFile, container) 
    #Transfer file downstreams
    outputData = container.GetOutputData()
    data = outputData["ACTGeometry"]
    actGeometry = data[0]
    actGeometry.TransferFile = geoFile

def buildjournal(container, extDir, projDir, geomFileName, dimension):
    """
        building journal to be run in WB
    """
    # Get the journal template from the extension directory.
    ExtAPI.Log.WriteMessage("Creating batch run journal.")    
    templateFilePath = System.IO.Path.Combine(extDir,'wbjnTemplate.tmpl')
    journalFilePath = System.IO.Path.Combine(projDir, "batchrun.wbjn")
    # Create substitution Path strings.
    resourceDir = Path.Combine(extDir, "resources")
    baseGeomFilePath = Path.Combine(resourceDir, "washer.agdb")
    exportGeomFilePath = Path.Combine(projDir, geomFileName)
    # replacing information about geometry in wbjn journal file
    ExtAPI.Log.WriteMessage("Substituting %GEOMFILEPATH% with = "+ baseGeomFilePath)
    ExtAPI.Log.WriteMessage("Substituting %EXPORTGEOMETRYFILEPATH% with = "+ exportGeomFilePath)
    ExtAPI.Log.WriteMessage("Substituting %INNERDIAMQTY% with = "+ str(dimension))
    exportGeomFilePath = exportGeomFilePath.replace("\\","\\\\\\\\")
    inStream = open(templateFilePath, 'r')
    buffer = inStream.read()
    buffer = buffer.replace('%GEOMFILEPATH%',baseGeomFilePath)
    buffer = buffer.replace('%EXPORTGEOMETRYFILEPATH%',exportGeomFilePath)
    buffer = buffer.replace('%INNERDIAMQTY%', str(dimension))
    outStream = open(journalFilePath, 'w')
    outStream.write(buffer)
    inStream.close()
    outStream.close()
    return(journalFilePath.replace("\\", "\\\\"))


def isInnerDiameterValid(task, property):
    """
        Check if the input is valid.
        The inner diameter has to be lower than Outer Diameter
    """
    quantity = property.Value
    if property.Value.GetType() == System.String:
        InnerDiameter = float(quantity.Split(" ")[0])
    else:
        InnerDiameter = property.Value.Value
    if InnerDiameter < 0:
        return False
    elif InnerDiameter > 34:
        return False
    else:
        return True
