using System;
using System.Collections;
using System.Collections.Generic;
using System.Runtime.Serialization;
using Ansys.Core;
using Ansys.Core.DataSpecs;

namespace ObjectTest
{
    //serializable class to satisfy ACT's save-resume process
    [TypeSpec(SpecType=typeof(MyClassSpec))]
    [Serializable]
    public class MyClass : ISerializable
    {
        public MyClass()
        {
            Name = "";
        }
        public MyClass(string name)
        {
            Name = name;
        }
        
        public MyClass(SerializationInfo info, StreamingContext context)
        {
            Name = (string) info.GetValue("Name", typeof(string));
        }
        
        [Spec]
        public string Name;
        
        public override string ToString()
        {
            return Name;
        }
        
        public void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue("Name", Name);
        }
    }
    
    //custom type spec to satisfy WB's save-resume process
    public class MyClassSpec : TypeSpec
    {
        public MyClassSpec()
            : base(typeof(MyClass))
        {
            AddBaseMyClassMembers();
        }
        
        public MyClassSpec(Type type)
            : base(type)
        {
            AddBaseMyClassMembers();
        }

        public MyClassSpec(Type type, params string[] typeHierarchy)
            : base(type, typeHierarchy)
        {
            AddBaseMyClassMembers();
        }

        public MyClassSpec(Func<object> creator)
            : base(creator)
        {
            AddBaseMyClassMembers();
        }

        public MyClassSpec(Func<object> creator, IEnumerable<string> typeHierarchy)
            : base(creator, typeHierarchy)
        {
            AddBaseMyClassMembers();
        }
        
        public MyClassSpec(params string[] typeHierarchy)
            : base(typeHierarchy)
        {
            AddBaseMyClassMembers();
        }
        
        public MyClassSpec(IEnumerable<string> typeHierarchy)
            : base(typeHierarchy)
        {
            AddBaseMyClassMembers();
        }

        //this class allows use to create an instance without
        //calling AddBaseMyClassMembers so that we do not 
        //get into an infinite loop when created through Activator.CreateInstance
        //in the SpecFactory
        internal MyClassSpec(bool temp)
            : base(typeof(MyClass))
        {
        }

        //override as we call AddMember when we create this spec
        //but AddMember is sometimes called later during the
        //SpecFactory creation process...which causes key conflicts.
        //avoid it my ignoring duplicates.
        public override void AddMember(string name, DataSpec spec)
        {
            if (Members.ContainsKey(name) == false)
            {
                base.AddMember(name, spec);
            }
        }
        
        //adds the discovered members to this class.
        //TypeSpec would usually do this via the EnsureBuilt and BuildFromType
        //calls - but we can't invoke those internal members...so do it now
        private void AddBaseMyClassMembers()
        {
            if (MemberCount == 0)
            {
                MyClassSpec tempSpec = new MyClassSpec(true);
                SpecFactory.BuildTypeSpec(typeof(MyClass), tempSpec);
                foreach (KeyValuePair<string, DataSpec> member in tempSpec.GetMembers())
                    AddMember(member.Key, member.Value);

            }
        }

        //If we deserialize the string and create a MyClass instance - just return it!
        public override object ConvertToImplementationType(object value, IExecutionContext context)
        {
            if (typeof(MyClass).IsInstanceOfType(value))
            {
                return value;
            }
            else
            {
                return base.ConvertToImplementationType(value, context);
            }
        }

        //deserialize the object based on the default string conversions
        //provided by StructSpec and the DataSpecs.  We know we will only
        //recieve the name-value combination, such as
        // {\"Name\":\"Value\"} since our only property is a string spec.
        public override object Deserialize(object value, IExecutionContext context)
        {
            if (value is string)
            {
                MyClass deserializedValue = null;
                Dictionary<string, object> dict = new Dictionary<string, object>();
                foreach (DictionaryEntry pair in ScriptSyntaxHelper.ConvertStringToDictionary((string)value))
                {
                    string key = (string)ChangeType(pair.Key, typeof(string));
                    string valueString = GetMemberIncludingPersistentOnly(key).Deserialize(pair.Value, context) as string;
                    deserializedValue = new MyClass(valueString);
                }

                return deserializedValue;
            }
            else
                return base.ConvertToImplementationType(value, context);
        }
    }
}