﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

using Ansys.Core.DataModel.DataObjects;
using Ansys.Core.DataModel.ProjectSystem;
using Ansys.Core.DataModel;
using Ansys.Core.Commands;

namespace Ansys.ScratchAddin.Commands
{
    [Command("DuplicateScratchComponent")]
    public partial class DuplicateScratchComponentCommand : ICommand
    {
        /// <summary>
        /// The cell's container base name.
        /// </summary>
        [Parameter]
        public string Name;

        /// <summary>
        /// the existing cell's container reference
        /// </summary>
        [Parameter]
        public DataContainerReference Container;

        /// <summary>
        /// System Properties to be able to access certain pieces of
        /// information to correctly replicate the cell
        /// </summary>
        [Parameter]
        public Dictionary<string, List<string>> AllSystemProperties;

        /// <summary>
        /// A Dictionary of upstream container to be able to correctly replicate
        /// upstream connections.
        /// </summary>
        [Parameter]
        public Dictionary<DataContainerReference, DataContainerReference> UpstreamContainerMap;

        /// <summary>
        /// The new "copy" of the existing container
        /// </summary>
        [Parameter(IsOptional = true)]
        public Output<DataContainerReference> CreatedContainer;


        public void Execute(IFullContext context)
        {
            //1.  Obtain the Source, or "Existing" Cell's DataObjectContainer
            DataObjectContainer sourceContainer = (DataObjectContainer)context.Project.GetContainerObject(Container, Addin.key);
            string containerType = sourceContainer.GetTypeString();

            //2.  Create the "Duplicate" DataObjectContainer (empty to start)
            DataObjectContainer container = new DataObjectContainer(containerType);

            //3.  Add the new DataObjectContainer to the Project
            DataContainerReference createdContainerRef = context.Project.AddContainer(container, Name, Addin.key);

            //4.  Copy over your existing cell's data objects, etc, using the Framework-Provided StandardDataObjectReplicator
            StandardDataObjectReplicator replicator = new StandardDataObjectReplicator(sourceContainer, container, context.Project);
            replicator.LockAndReplicateAll(context, UpstreamContainerMap);

            //(OPTIONAL) Duplicate file associations.
            DataReferenceSet fileRefs = Ansys.Core.FileManagement.Queries.GetFilesForContainerQuery.InvokeAndWait(context, Container);

            foreach (DataReference file in fileRefs)
            {
                //associate each file with the new, duplicated container.
                Ansys.Core.FileManagement.Commands.AssociateFileWithContainerCommand.InvokeAndWait(context, file, createdContainerRef);
            }

            //5.  Duplicate any Parameters that your existing cell may have defined
            Ansys.ParameterManager.Commands.DuplicateParametersForContainerCommand.InvokeAndWait(context, Container, createdContainerRef);

            //6.  Check for proper state and adjust where needed
            AdjustDuplicateComponentState(context, Container, createdContainerRef);

            //7.  Set the new container's unit system if you have defined one for the existing cell.
            DataReference containerSystemReference = Ansys.Units.Queries.GetContainerUnitSystemQuery.InvokeAndWait(context, Container);
            string unitSystemName;
            if (containerSystemReference != null)
            {
                using (context.ContainerReadLock(containerSystemReference.Container))
                {
                    unitSystemName = (string)context.Project.GetProperty(containerSystemReference, "UnitSystemName");
                }
                Ansys.Units.Commands.RegisterContainerUnitSystemCommand.InvokeAndWait(context, createdContainerRef, unitSystemName);
            }

            //8.  Copy over any input files needed for the component
            string sourceSystemID = Ansys.ProjectSchematic.Queries.GetUniqueSystemDirectoryNameForContainerQuery.InvokeAndWait(context, Container);
            string sourceDirectory = context.Project.FileManager.GetActiveAddinDirectoryPath(sourceSystemID, "Scratch");
            string newSystemId = AllSystemProperties[Ansys.ProjectSchematic.SystemPropertyNames.UniqueSystemDirectoryName][0];
            string newDirectory = context.Project.FileManager.GetActiveAddinDirectoryPath(newSystemId, "Scratch");
            Ansys.Utilities.FileSystem.CopyDirectory(sourceDirectory, newDirectory, true, Ansys.Utilities.DuplicateFileResolvePolicy.Overwrite);

            //9.  Set the Output parameter to the new, "duplicated" container.
            Ansys.Core.Commands.Output.SetIfNotNull(CreatedContainer, createdContainerRef);

        }

        /// <summary>
        /// Method to handle state checking upon component duplication
        /// </summary>
        /// <param name="context"></param>
        /// <param name="sourceContainer"></param>
        /// <param name="targetContainer"></param>
        internal static void AdjustDuplicateComponentState(
            IFullContext context, DataContainerReference sourceContainer,
            DataContainerReference targetContainer)
        {
            //State adjustement for duplicate component:
            //these components always copy generated data, therefore 
            //if the source component has generated output (state == upToDate or Interrupted)   -> duplication calls SignalOutputsGenerated
            //if the source component is not in need to refresh the duplicated won't be either  -> duplication calls SignalRefreshed
            //A second pass is needed to adjust for Modified state of the Source Container -> duplication calls SignalModified
            DataObjectContainer createdContainerobj = (DataObjectContainer)context.Project.GetContainerObject(targetContainer, Addin.key);
            bool isRefreshable = true;
            bool isUpdateable = true;
            bool isModified = false;
            foreach (DataReference cell in Ansys.ProjectSchematic.Queries.GetComponentsForContainerQuery.InvokeAndWait(context, sourceContainer, true))
            {
                //Check if any of the system's components are out-of-date (needs refreshed) or needs updated.
                Ansys.ProjectSchematic.State state = Ansys.ProjectSchematic.Queries.GetComponentStateQuery.InvokeAndWait(context, cell).State;
                if (state == Ansys.ProjectSchematic.State.OutOfDate)
                    isRefreshable = false;
                if ((state == Ansys.ProjectSchematic.State.Incomplete) ||
                    (state == Ansys.ProjectSchematic.State.EditRequired) ||
                    (state == Ansys.ProjectSchematic.State.Unfulfilled))
                    isUpdateable = false;
            }
            using (context.ContainerWriteLock(targetContainer))
            {
                //send the proper signals so that WB knows what state the cells are in or if the containers
                //have generated its outputs for downstream consumption
                if (isRefreshable)
                    //createdContainerobj.SignalRefreshed();
                    if (isUpdateable)
                        createdContainerobj.SignalOutputsGenerated();
            }

            foreach (DataReference cell in Ansys.ProjectSchematic.Queries.GetComponentsForContainerQuery.InvokeAndWait(context, sourceContainer, true))
            {
                //check if the system has a modified component
                Ansys.ProjectSchematic.State state = Ansys.ProjectSchematic.Queries.GetComponentStateQuery.InvokeAndWait(context, cell).State;
                if (state == Ansys.ProjectSchematic.State.Modified)
                    isModified = true;
            }
            using (context.ContainerWriteLock(targetContainer))
            {
                //if a system's components is modified, singal this to WB.
                if (isModified)
                    context.Project.SignalModified(createdContainerobj.ContainerReference);
            }
        }
    }
}
