﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using System.IO;

using Ansys.UI;
using Ansys.UI.Interfaces;
using Ansys.Core.DataModel.ProjectSystem;
using Ansys.Core.DataModel.DataObjects;
using Ansys.Core.DataModel;
using Ansys.Core.Commands;
using Ansys.ProjectSchematic;
using Ansys.ProjectSchematic.Data;
using Ansys.Utilities;

using Ansys.ScratchAddin.Data;

namespace Ansys.ScratchAddin
{
    public class Addin : Ansys.Core.Addins.PersistableAddinBase, IDefineGui
    {
        //The internal key that allows DataObjectContainer retrieval from within this Addin
        internal static object key = new object();

        #region AddinBase required overrides

        /// <summary>
        /// Load the DataModel/Core parts of the Addin...called during Workbench initialization
        /// </summary>
        /// <param name="context">the context</param>
        public override void Load(Ansys.Core.Addins.AddinLoadContext context)
        {
            //register all of the commands/queries defined in the addin
            context.CommandManager.RegisterAllFromExecutingAssembly();

            //setup the component and system templates for this Addin
            CreateTemplates(context);
        }

        public override void Unload(Ansys.Core.Addins.AddinUnloadContext context)
        {
        }

        #endregion

        #region IDefineGui Members

        /// <summary>
        /// Load the GUI parts of the Addin...called during Workbench initialization
        /// </summary>
        /// <param name="context"></param>
        public void Load(Ansys.UI.GuiDefineContext context)
        {
            //register all GUI operations defined in this Addin
            context.RegisterAllFromExecutingAssembly();

            //set the image path for the Addin (for cell icons, etc)
            string imgdir = Path.Combine(Path.GetDirectoryName(Path.GetDirectoryName(
                    Path.GetDirectoryName(
                        System.Reflection.Assembly.GetExecutingAssembly().Location))), "Images");

            if (!Directory.Exists(imgdir) &&
                 Directory.Exists(Path.Combine(Path.GetDirectoryName(
                    Path.GetDirectoryName(imgdir)), "Images")))
            {
                imgdir = Path.Combine(Path.GetDirectoryName(Path.GetDirectoryName(imgdir)), "Images");
            }
            context.RegisterImageDirectory(imgdir);
        }

        #endregion

        /// <summary>
        /// Creates the Component and System Templates for the presentation side of the Addin
        /// </summary>
        /// <param name="context"></param>
        internal void CreateTemplates(Ansys.Core.Addins.AddinLoadContext context)
        {
            //setting up all of the possible combinations of inputs for the Solution component.
            List<RequiredInput> Inputs_Empty = new List<RequiredInput>();
            List<RequiredInput> Inputs_Scratch = new List<RequiredInput>(new RequiredInput[] 
            {
                new RequiredInput(Core.AddinConstants.ScratchDataTypeString)
            });

            //Registering the component template for the Producer
            context.TemplateRepository.AddComponentTemplate("ScratchTemplate")
                .SetRequiredInputTypes(new IEnumerable<RequiredInput>[] { 
                    Inputs_Empty })
                .SetCreatingCommand(typeof(Commands.CreateScratchComponentCommand)) //define the command that will create the component
                .SetRefreshCommand(typeof(Commands.RefreshScratchComponentCommand))
                .SetUpdateCommand(typeof(Commands.UpdateScratchComponentCommand)) //define the command that will handle component updating
                .SetDuplicateContainerCommand(typeof(Commands.DuplicateScratchComponentCommand))
                .SetComponentPropertyDataQuery(typeof(Queries.GetScratchComponentPropertiesQuery)) //define the command that will provide the properties to display in the Property View for this component
                .SetComponentStatusQuery(typeof(Queries.GetScratchComponentStatusQuery)) //define the query that will notify what the Component's state should be
                .SetCreateAllProvidesConnectionsToNewSystem(true)
                .SetIsShareable(false)
                .SetOutputTypes(new string[] { Core.AddinConstants.ScratchDataTypeString }) //set the output type that the component will generate upon a successful update
                .SetDisplayName(""); //set the UI string that the user will see as the component's name

            //Registering the System template for the Addin
            context.TemplateRepository.AddSystemTemplate("ScratchTemplate")
                .SetToolboxGroup("Scratch") //Set which Toolbox group the system will be added to
                .SetComponentTemplateNames(new string[] { "ScratchTemplate" })
                .SetComponentNames(new string[] { "Scratch" })
                .SetSystemTypeAbbreviation("ScratchEX") //Set the System Type Abbreviation for this system
                .SetDefaultSystemName("Scratch Example") //Set the default, user-visible name for the system
                .SetToolboxDisplayText("Scratch Example") //Set the system name that will be displayed in the Toolbox
                .SetImage("ScratchImage"); //Set the .png name for the image that will be displayed in the Toolbox
        }

        public override Version Version
        {
            get
            {
                return new Version(1, 0);
            }
        }

        //NOTE:  The method overrides below are strictly for saving and opening project with this system inside them...they do not affect Data Transfer.
        //PersistableAddinBase override of Save - to facilitate the proper Saving of the Addin and its files
        public override void Save(Ansys.Core.Persistence.IStorageObjectWriter storage, Ansys.Core.Persistence.PersistenceContext context, Ansys.Core.Persistence.PersistenceScope scope)
        {
            base.Save(storage, context, scope);
        }

        //PersistableAddinBase override of GetContainerPersistenceScope - sets at which scope the Addin's persistent data will reside...here we say it will be in the Design Point scope
        public override Ansys.Core.Persistence.PersistenceScope GetContainerPersistenceScope(string containerType)
        {
            return Ansys.Core.Persistence.PersistenceScope.DesignPoint;
        }

        //PersistableAddinBase override of OnContainerLoad - What to do when our Addin's container is loaded from a saved project...we must re-key it to be able to properly access it after the load.
        public override void OnContainerLoad(IDataContainer container, DataContainerReference reference, Ansys.Core.Persistence.PersistenceContext context)
        {
            context.ReKeyContainer(reference, Addin.key);
        }

        //PersistableAddinBase override of CreateContainer - How to handle the creation of a container for this Addin during a project load.
        public override IDataContainer CreateContainer(string containerType)
        {
            return new DataObjectContainer(containerType);
        }

        //PersistableAddinBase override of Migrate - Handle any entity migrations needed from an older version to a newer version....maintains the ability for bakwards compatibility.
        public override void Migrate(Ansys.Core.Persistence.PersistenceContext context, Ansys.Core.Persistence.PersistenceScope scope)
        {
            //migration code would go here...
            base.Migrate(context, scope);
        }
    }
}
