# ----------------------------------------------------------
# Copyright (C) 2019 by
# ANSYS Switzerland GmbH
# www.ansys.com
#
# Author(s): O.Koenig
# ----------------------------------------------------------
#
# Example script to setup a DCS project with Python

import logging
import os
import random

from ansys.dcs.client import DCSError
from ansys.dcs.client.dps import Client
from ansys.dcs.client.dps import (Configuration, DesignPoint, File,
                                           FitnessDefinition, Project,
                                           SuccessCriteria)

log = logging.getLogger(__name__)

def main():
    """Create a DCS project consisting of an ANSYS APDL beam model
    of a tubular steel trellis motorbike-frame.

    After creating the project configuration, 10 design points with randomly 
    chosen parameter values are created and set to pending.

    For further details about the model and its parametrization, see e.g.
    "Using Evolutionary Methods with a Heterogeneous Genotype Representation 
    for Design Optimization of a Tubular Steel Trellis Motorbike-Frame", 2003
    by U. M. Fasel, O. Koenig, M. Wintermantel and P. Ermanni.
    """

    log.debug("=== DCS connection")
    client = Client(dcs_url="https://127.0.0.1/dcs", username="dcadmin", password="dcadmin")

    log.debug("=== Project")
    proj=Project(id="mapdl_motorbike_frame", display_name="MAPDL Motorbike Frame", priority=1, active=True)
    proj=client.create_project(proj, replace=True)

    log.debug("=== Files")
    cwd=os.path.dirname(__file__)
    files=[]    
    files.append (File( name="mac",evaluation_path="motorbike_frame.mac",
                        type="text/plain", src=os.path.join(cwd, "motorbike_frame.mac") ) )
    files.append( File( name="results", evaluation_path="motorbike_frame_results.txt", 
                        type="text/plain", src=os.path.join(cwd, "motorbike_frame_results.txt") ) )
    files.append( File( name="img", evaluation_path="file000.jpg", type="image/jpeg") )
    files.append( File( name="img2", evaluation_path="file001.jpg", type="image/jpeg") )
    files.append( File( name="out", evaluation_path="file.out", type="text/plain") )
    files=proj.create_files(files)
    mac_file=files[0]
    result_file=files[1]

    log.debug( "=== Configuration with simulation workflow and parameters")
    cfg=Configuration(name="Configuration.1", active=True, )

    # Input params: Dimensions of three custom tubes
    float_input_params=[]
    for i in range(1,4):
        pd=cfg.add_float_parameter_definition(name='tube%i_radius' %i, lower_limit=4.0, upper_limit=20.0,default=12.0 )
        cfg.add_parameter_location(key_string='radius(%i)' % i, tokenizer="=", parameter_definition_name=pd.name, file_id=mac_file.id)
        float_input_params.append(pd)

        pd=cfg.add_float_parameter_definition(name='tube%i_thickness' %i,lower_limit=0.5, upper_limit=2.5, default=1.0 )
        cfg.add_parameter_location(key_string='thickness(%i)' % i, tokenizer="=", parameter_definition_name=pd.name, file_id=mac_file.id)
        float_input_params.append(pd)
        
    # Input params: Custom types used for all the different tubes of the frame
    str_input_params=[]
    for i in range(1,22):
        pd=cfg.add_string_parameter_definition(name="tube%s" %i, default="1", value_list=["1","2","3"] )
        cfg.add_parameter_location(key_string='tubes(%i)' % i, tokenizer="=", parameter_definition_name=pd.name, file_id=mac_file.id)
        str_input_params.append(pd)

    # Output Parames
    for pname in ["weight", "torsion_stiffness", "max_stress"]:
        pd=cfg.add_float_parameter_definition(name=pname)
        cfg.add_parameter_location(key_string=pname, tokenizer="=", parameter_definition_name=pd.name, file_id=result_file.id)
    
    # Process step
    cfg.add_process_step(   name="MAPDL_run", 
                            application_name="ANSYS Mechanical APDL",
                            application_version="20.1",
                            execution_command="%executable% -b -i %file:mac% -o file.out",
                            max_execution_time=20.0,
                            cpu_core_usage=1,
                            execution_level=0,
                            memory=250,
                            disk_space=5,
                            num_trials=3,
                            input_file_ids=[f.id for f in files[:1]], 
                            output_file_ids=[f.id for f in files[1:]], 
                            success_criteria= SuccessCriteria(
                                return_code=0,
                                expressions= ["values['tube1_radius']>=4.0", "values['tube1_thickness']>=0.5"],
                                required_output_file_ids=[ f.id for f in files[2:] ],
                                require_all_output_files=False,
                                require_all_output_parameters=True
                            ) )

    # Design point files
    cfg.create_design_point_file_ids = [f.id for f in files[2:]]

    # Fitness definition
    fd = FitnessDefinition(error_fitness=10.0)
    fd.add_fitness_term(name="weight", type="design_objective", weighting_factor=1.0,
                        expression="map_design_objective( values['weight'], 7.5, 5.5)")
    fd.add_fitness_term(name="torsional_stiffness", type="target_constraint", weighting_factor=1.0,
                    expression="map_target_constraint( values['torsion_stiffness'], 1313.0, 5.0, 30.0 )" )
    fd.add_fitness_term(name="max_stress", type="limit_constraint", weighting_factor=1.0,
                    expression="map_limit_constraint( values['max_stress'], 451.0, 50.0 )") 
    cfg.fitness_definition =fd
 
    # Create configuration in project
    cfg=proj.create_configurations([cfg])[0]

    log.debug( "=== Design points")
    dps=[]
    for i in range(10):
        values = { p.name : p.lower_limit + random.random()*(p.upper_limit-p.lower_limit)  for p in float_input_params }
        values.update({ p.name: random.choice(p.value_list) for p in str_input_params})
        dps.append( DesignPoint( name=f"DesignPoint.{i}", values=values, eval_status="pending") )
    dps=cfg.create_design_points(dps)


if __name__ == "__main__":

    logger = logging.getLogger()
    logging.basicConfig(format='[%(asctime)s | %(levelname)s] %(message)s', level=logging.DEBUG)

    try:
        main()
    except DCSError as e:
        log.error(str(e))
