# ----------------------------------------------------------
# Copyright (C) 2019 by
# ANSYS Switzerland GmbH
# www.ansys.com
#
# Author(s): O.Koenig
# ----------------------------------------------------------
#
# Example script to setup a DCS project with Python

import logging
import os
import random

from ansys.dcs.client import DCSError
from ansys.dcs.client.dps import (Client, Configuration, DesignPoint, File,
                                  FitnessDefinition, Project, SuccessCriteria, Licensing)
from ansys.dcs.client import __external_version__ as ansys_version

log = logging.getLogger(__name__)

def create_dcs_project(client, proj_id, num_design_points=20):
    """Create a DCS project consisting of an ANSYS APDL beam model
    of a tubular steel trellis motorbike-frame.

    After creating the project configuration, 10 design points with randomly 
    chosen parameter values are created and set to pending.

    For further details about the model and its parametrization, see e.g.
    "Using Evolutionary Methods with a Heterogeneous Genotype Representation 
    for Design Optimization of a Tubular Steel Trellis Motorbike-Frame", 2003
    by U. M. Fasel, O. Koenig, M. Wintermantel and P. Ermanni.
    """

    log.debug("=== Project")
    proj = Project(id=proj_id, display_name="MAPDL Motorbike Frame", priority=1, active=True)
    proj = client.create_project(proj, replace=True)

    log.debug("=== Files")
    cwd = os.path.dirname(__file__)
    files = []
    files.append(File(name="mac", evaluation_path="motorbike_frame.mac",
                      type="text/plain", src=os.path.join(cwd, "motorbike_frame.mac")))
    files.append(File(name="results", evaluation_path="motorbike_frame_results.txt",
                      type="text/plain", src=os.path.join(cwd, "motorbike_frame_results.txt")))
    files.append(File(name="out", evaluation_path="file.out", type="text/plain", collect=True, monitor=True))
    files.append(File(name="img", evaluation_path="**.jpg", type="image/jpeg", collect=True))
    files.append(File(name="err", evaluation_path="file*.err", type="text/plain", collect=True, monitor=True))

    # Alternative, not recommended way, will collect ALL files matching file.*
    #files.append( File( name="all_files", evaluation_path="file.*", type="text/plain") )

    files = proj.create_files(files)
    file_ids = {f.name: f.id for f in files}

    log.debug("=== Configuration with simulation workflow and parameters")
    cfg = Configuration(name="Configuration.1", active=True)

    # Input params: Dimensions of three custom tubes
    float_input_params=[]
    for i in range(1,4):
        pd=cfg.add_float_parameter_definition(name='tube%i_radius' %i, lower_limit=4.0, upper_limit=20.0,default=12.0 )
        cfg.add_parameter_location(key_string='radius(%i)' % i, tokenizer="=", parameter_definition_name=pd.name, file_id=file_ids["mac"])
        float_input_params.append(pd)

        pd=cfg.add_float_parameter_definition(name='tube%i_thickness' %i,lower_limit=0.5, upper_limit=2.5, default=1.0 )
        cfg.add_parameter_location(key_string='thickness(%i)' % i, tokenizer="=", parameter_definition_name=pd.name, file_id=file_ids["mac"])
        float_input_params.append(pd)
        
    # Input params: Custom types used for all the different tubes of the frame
    str_input_params=[]
    for i in range(1,22):
        pd=cfg.add_string_parameter_definition(name="tube%s" %i, default="1", value_list=["1","2","3"] )
        cfg.add_parameter_location(key_string='tubes(%i)' % i, tokenizer="=", parameter_definition_name=pd.name, file_id=file_ids["mac"])
        str_input_params.append(pd)

    # Output Parames
    for pname in ["weight", "torsion_stiffness", "max_stress"]:
        pd=cfg.add_float_parameter_definition(name=pname)
        cfg.add_parameter_location(key_string=pname, tokenizer="=", parameter_definition_name=pd.name, file_id=file_ids["results"])

    # Collect some runtime stats from MAPDL out file
    pd=cfg.add_float_parameter_definition(name="mapdl_elapsed_time_obtain_license")
    cfg.add_parameter_location(key_string="Elapsed time spent obtaining a license", tokenizer=":", parameter_definition_name=pd.name, file_id=file_ids["out"])
    pd=cfg.add_float_parameter_definition(name="mapdl_cp_time")
    cfg.add_parameter_location(key_string="CP Time      (sec)", tokenizer="=", parameter_definition_name=pd.name, file_id=file_ids["out"])
    pd=cfg.add_float_parameter_definition(name="mapdl_elapsed_time")
    cfg.add_parameter_location(key_string="Elapsed Time (sec)", tokenizer="=", parameter_definition_name=pd.name, file_id=file_ids["out"])

    # Process step
    cfg.add_process_step(   name="MAPDL_run", 
                            application_name="ANSYS Mechanical APDL",
                            application_version=ansys_version,
                            execution_command="%executable% -b -i %file:mac% -o file.out -np %num_cores%",
                            max_execution_time=50.0,
                            cpu_core_usage=1.0,
                            execution_level=0,
                            memory=250,
                            disk_space=5,
                            num_trials=1,
                            input_file_ids=[f.id for f in files[:1]], 
                            output_file_ids=[f.id for f in files[1:]], 
                            success_criteria= SuccessCriteria(
                                return_code=0,
                                expressions= ["values['tube1_radius']>=4.0", "values['tube1_thickness']>=0.5"],
                                required_output_file_ids=[ file_ids["results"] ],
                                require_all_output_files=False,
                                require_all_output_parameters=True
                            ),
                            licensing = Licensing(enable_shared_licensing=False) # Shared licensing disabled by default
                        )

    # For demonstration purpose we also define some parameter replacements that refer to process step properties
    cfg.add_parameter_location(key_string='name', tokenizer="=", string_quote="'", process_step_property="name", file_id=file_ids["mac"])
    cfg.add_parameter_location(key_string='application_name', tokenizer="=", string_quote="'", process_step_property="application_name", file_id=file_ids["mac"])
    cfg.add_parameter_location(key_string='num_cores', tokenizer="=", process_step_property="num_cores", file_id=file_ids["mac"])
    cfg.add_parameter_location(key_string='cpu_core_usage', tokenizer="=", process_step_property="cpu_core_usage", file_id=file_ids["mac"])

    # Fitness definition
    fd = FitnessDefinition(error_fitness=10.0)
    fd.add_fitness_term(name="weight", type="design_objective", weighting_factor=1.0,
                        expression="map_design_objective( values['weight'], 7.5, 5.5)")
    fd.add_fitness_term(name="torsional_stiffness", type="target_constraint", weighting_factor=1.0,
                    expression="map_target_constraint( values['torsion_stiffness'], 1313.0, 5.0, 30.0 )" )
    fd.add_fitness_term(name="max_stress", type="limit_constraint", weighting_factor=1.0,
                    expression="map_limit_constraint( values['max_stress'], 451.0, 50.0 )") 
    cfg.fitness_definition =fd
    
    # Create configuration in project
    cfg=proj.create_configurations([cfg])[0]

    log.debug( f"=== Create {num_design_points} design points" )
    dps=[]
    for i in range(num_design_points):
        values = { p.name : p.lower_limit + random.random()*(p.upper_limit-p.lower_limit)  for p in float_input_params }
        values.update({ p.name: random.choice(p.value_list) for p in str_input_params})
        dps.append( DesignPoint( name=f"DesignPoint.{i}", values=values, eval_status="pending") )
    dps=cfg.create_design_points(dps)

    return proj

if __name__ == "__main__":

    logger = logging.getLogger()
    logging.basicConfig(
        #format='[%(asctime)s | %(levelname)s] %(message)s', 
        format='%(message)s', 
        level=logging.DEBUG)

    try:
        log.debug("=== DCS connection")
        client = Client(dcs_url="https://127.0.0.1/dcs", username="dcadmin", password="dcadmin")
        proj=create_dcs_project(client=client, proj_id="mapdl_motorbike_frame", num_design_points=500)

    except DCSError as e:
        log.error(str(e))
